#! /usr/bin/env python
# -*- coding: utf-8 -*-
"""
BX1200ctrl.py

Simple control program for BX-1200 Antenna Tuner
================================================
Supports setting frequency, monitoring TX power and
tracking of frequency from hamblib rigctld

This program comes with absolute no warranty.
Usage is own your own risk.

(C) 2017, 2018, DG7BBP, Jens Rosebrock

"""

_version = "1.04"

import argparse
import collections
import serial
import socket
import sys
import time
import math



EOT = '\x04'
LF = '\x0A'
_verbose_level = -1


def log(level, msg):
    global _verbose_level
    if level <= _verbose_level:
        sys.stdout.write("%s\n" % msg)

def calc_swr(r, x):
    """
    :param r: float real resistance
    :param x: float reatance

    :returns: float swr againt 50 Ohm
    """
    z0 = 50.0
    z = math.sqrt(r*r + x*x)
    p_abs = abs((z - z0) / (z + z0))
    swr = (1.0 + p_abs) / (1.0 - p_abs)
    return swr

def show_error(msg):
    sys.stderr.write("%s\n" % msg)


def show_exception(e):
    msg = e.message
    if not msg:
        msg = e.strerror
    if not msg:
        msg = "%s" % e
    show_error(msg)


class HamLibConnection(object):
    """
    Simple class for requesting frequncy from rigctld 
    """

    def __init__(self, host_port):
        self._host_port = host_port
        self._socket = None
        
    def __enter__(self):
        """
        Allow with statement
        """
        self.connect()
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        """
        """
        self.disconnect()        

    def connect(self, retry=1):
        """
        Connect to rightld
        :raises socket.error
        """
        addr_data = self.url_to_host(self._host_port, 4532)
        if len(addr_data)==2:
            s = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        else:
            s = socket.socket(socket.AF_INET6, socket.SOCK_STREAM)
        last_err = None
        for i in range(retry):
            last_err = None
            try:
                s.connect(addr_data)
                break
            except (socket.error,) , e:
                last_err = e
                time.sleep(1)
        if last_err is  None:
            self._socket = s
        else:
            raise last_err

    def disconnect(self):
        """
        Disconnect from rigtld
        """
        if self._socket is not None:
            self._socket.close()

    def url_to_host(self, host_port, default_port):
        """
        get hostname, ip address from host_port string like:
        <hostname>[:<port>]
        <ipv4_addr>[:<port>]
        \[<ipv6_addr>\][:<port>]
        Port is optional
        """
        splitted = host_port.split(":")
        port_str = None
        is_ipv6 = False
        if len(splitted) == 1:
            # ip v4/hostname no port
            host = host_port
        elif len(splitted) == 2:
            # ip v4/hostname given port
            host = splitted[0]
            port_str = splitted[1]
        else:
            log(3, "url_to_host: IPV6 %s" % (splitted,))
            is_ipv6 = True
            # some kind of ip6
            if splitted[-1].endswith("]"):
                # no port
                log(3,"url_to_host: NO PORT")
                host = host_port
                if host.startswith("[") and host.endswith("]"):
                    host = host[1:-1]
            else:
                port = splitted[-1]
                host  = ":".join(splitted[:-1])
                if host.startswith("[") and host.endswith("]"):
                    host = host[1:-1]
        if port_str is None or not port_str.isdigit():
            port = default_port
        else:
            port = int(port_str)
        log(3, "url_to_host: HOST %s PORT: %s %s" % (host, port, is_ipv6))
        if is_ipv6:
            return host, port , 0, 0
        else:
            return host, port

    def read_frequency(self):
        """
        Simple read function for reading the current frequency in Hz
        from rigctld. No special error handling is supplied.

        :return: None if error else  float frequency in Hz
        """
        result = None
        if self._socket:
            self._socket.sendall("f"+chr(13))
            data = self._socket.recv(1000)
            log(3, "hamlib_read_frequency: %s" % data)
            if data:
                if not data.startswith("RPRT"):
                    result = float(data.strip())
        return result


class BX1200Status(object):
    """
    Represents the result status of various BX-1200 commands
    Some commands are sending the same keyword twice in a message,
    tht's why we can't use a simple dict.
    """

    def __init__(self, last_command, result_data = None):
        """
        """
        self.cmd = last_command # Last Command, Supported all, (Frq!, RdE!)
        self.frq = None
        self.vcc = None
        self.tmp = None
        self.uc2 = None
        self.wrn = None
        self.eta = None
        self.fwd = None
        self.rin = None
        self.xin = None
        self.fau = None
        self.cha = None
        self.mml = None
        self.mmh = None
        self.chl = None
        self.chh = None
        self.mrk = None
        self.atc = None
        self.bnk = None

        # ab jetzt frequenzautomatik
        self.facha = None
        self.factu = None
        self.faltu = None
        self.faatc = None
        self.fabnk = None

        self.Attr2Var = {"Frq": (int, "frq"),
                          "Vcc":  (float, "vcc"),
                          "Tmp": (float, "tmp"),
                          "UC2": (float, "uc2"),
                          "Wrn": (int, "wrn"),
                          "eta": (float, "eta"),
                          "FWD": (float, "fwd"),
                          "Rin": (float, "rin"),
                          "Xin": (float, "xin"),
                          "FAu": (int, "fau"),
                          "Cha": (int, "cha"),
                          "MmL": (int, "mml"),
                          "MmH": (int, "mmh"),
                          "ChL": (int, "chl"),
                          "ChH": (int, "chh"),
                          "Mrk": (int, "mrk"),
                          "ATC": (int, "atc"),
                          "Bnk": (int, "bnk")
                          }

        self.FaAttr2Var = {
                           "Cha": (int, "facha"),
                           "Ctu": (int, "factu"),
                           "Ltu": (int, "faltu"),
                           "ATC": (int, "faatc"),
                           "Bnk": (int, "fabnk")
                           }
        if result_data is not None:
            self.scan_result(result_data)

    def scan_result(self, msg):
        """
        reads msg string of key vaulue pairs which are teminated by a LF.
        """
        if msg:
            split_msg = msg.split(LF)
            recno = 0
            for record in split_msg:
                if record:
                    split_rec = record.split("=")
                    if len(split_rec) == 2:
                        # print record
                        key = split_rec[0]
                        val = split_rec[1]
                        if recno < 15:
                            destination = self.Attr2Var.get(key)
                            if destination is None:
                                destination = self.FaAttr2Var.get(key)
                        else:
                            destination = self.FaAttr2Var.get(key)
                        if destination is not None:
                            log(3, "setting attribute %s = %s" % (destination[1], val))
                            setattr(self, destination[1], destination[0](val))
                        else:
                            show_error("ERROR unknown result: '%s'='%s'" % (key, recno))
                    else:
                       show_error("ERROR unexpected result: '%s' " % msg)
                    recno += 1

    def show(self):
        sys.stdout.write("%s\n" % self.to_string())

    def to_string(self):

        s_list = []
        for t in self.Attr2Var.values():
            if getattr(self, t[1]) is not None:
                s_list.append("%s: %s" % (t[1], getattr(self, t[1])))
        s = "\n".join(s_list)
        return s


class CommunicationError(Exception):
    pass


class BX1200Error(Exception):
    pass


class BX1200Ctrl(object):
    """
    Class for controlling BX-1200 Antenna tuner
    """

    BAUDRATE = 57600

    def __init__(self, comport):
        self._comport = comport
        self._connection = None
        self._over_mode = False
        
    def __enter__(self):
        """
        Allow with statement
        """
        self.connect()
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        """
        """
        self.disconnect()

    def connect(self):
        """
        :raises: CommunicationError
        """
        if self._connection is not None:
                self.disconnect()
        self._connection = serial.Serial(self._comport, self.BAUDRATE, timeout=0.6)
        self.send_cmd("COM?")
        self._over_mode = False
        result = self._read_transmission_from_bx1200()
        if not result or result.strip() != "conn":
            self.disconnect()
            raise CommunicationError("COM port %s is not valid or no tuner found!" % self._comport)
        result = self.read_all_parameter()
        if result:
            if result.fau != 0:
                show_error("Frequenzautomatik wurde ausgeschaltet!")
                self.enable_frequency_auto(False)


    def disconnect(self):
        if self._connection is not None:
            self._over_mode = False
            self._connection.close()
            self._connection = None

    def send_cmd(self, cmd):
        """
        Send command to BX-1200
        handles _over_mode_status
        """
        if self._connection is not None:
            log(3, "bx1200_send_cmd: %s" %cmd)
            tx_data = "%s%s" % (cmd, LF)
            ret = self._connection.write(tx_data)
            self._connection.flush()
            self._over_mode = cmd == "Over"

    def set_freqency(self, f, set_relais=True):
        """
        :param f: float. Frequency in hz
        :param set_relais: boolean. Send EdE! cmd after reading frequency.
                                    Should only be used if no overload condition
                                    for switching the relais is present
        : returns BX1200Status
        """
        result = None
        if f <= 30e6 and f > 100e3:
            f_khz = int(f / 1000)
            cmd = "Frq!%05d" % f_khz
            self.send_cmd(cmd)
            result_data = self._read_transmission_from_bx1200()
            # print result_data
            if result_data:
                result = BX1200Status("Frq!", result_data)
                if set_relais:
                    self.send_cmd("RdE!")
                    result_data = self._read_transmission_from_bx1200()
                    if result_data and result_data != "OvLoad" + EOT:
                        result.scan_result(result_data)
                    else:
                        raise BX1200Error(result_data if result_data else "No answer")
            else:
                 raise BX1200Error("No answer (out of sync?")
        else:
            raise ValueError
        return result

    def enable_frequency_auto(self, on):
        """
        :param on: Boolean
        :raise BX1200Error
        """
        val = 1 if on else 0
        cmd = "FAu%s" % val
        self.send_cmd(cmd)
        result_data = self._read_transmission_from_bx1200()
        if result_data:
            result = BX1200Status("FAu", result_data)
            if result.fau != val:
                BX1200Error("Setting Frequency automic failed!")


    def read_all_parameter(self):
        """
        Reads parameters from all? command
        :return: None if error (no answer) or BX1200Status
        """
        self.send_cmd("all?")
        result_data = self._read_transmission_from_bx1200()
        if result_data:
            result = BX1200Status("all?", result_data)
            return result
        else:
            BX1200Error("No answer (out of sync?")


    def send_over(self):
        """
        Sends Over Cmd for waiting for activation from BX-1200
        """
        if not self._over_mode:
            self.send_cmd("Over")

    def query_tx_data(self):
        """
        """
        result_data = self._read_transmission_from_bx1200()
        # print result_data
        if result_data:
            result = BX1200Status("Over", result_data)

            if result.fwd > 0:
                return self.read_all_parameter()

    def get_tx_data_iterator(self):
        """
        return status until no more tx power is detected
        assumes that bx12200 is in over mode
        """
        log(3,"get_tx_data_iterator: Begin ***")
        # wait a little bit after ove
        time.sleep(0.5)
        result_data = self._read_transmission_from_bx1200()
        if result_data:
            result = BX1200Status("Over", result_data)
            log(3,"get_tx_data_iterator: Result Data")
            while result and result.fwd is not None and result.fwd > 0:
                if result.rin is not None:
                    # !st answer after conatins no additional infos
                    yield result
                # this routine waits time 1 s for an ansswer so we
                # don't need an additional sleep.
                result = self.read_all_parameter()
        log(3,"get_tx_data_iterator: End **")


    def _read_transmission_from_bx1200(self):
        """
        reads until EOT is received includes EOT in message

        :returns None if no data was received until timeout
                 or datagram time exceeds 2.0 seconds else
                 rerzurn character string
        """
        result = None
        if self._connection is not None:
            result_list = []
            msg_timeout = 2.0
            first_char_time = None
            start_time = time.time()

            last_char = self._connection.read(1)
            while (last_char is not None and result is None):
                if not result_list:
                    #  reset over mode in case of an unkown result.
                    #  sometimes the message is incomplete
                    self._over_mode = False
                    first_char_time = time.time()
                    #if timeout and (time.time() - start_time) > timeout:
                    #    log(3, "bx1200_read: timeout reached")
                     #   result = None
                     #   break
                result_list.append(last_char)
                last_char = self._connection.read(1)
                if last_char == EOT:
                    result = "".join(result_list)
                elif first_char_time and (time.time()- first_char_time) > msg_timeout:
                    log(3, "bx1200_read: msg_timeout reached")
                    result = None
                    break
            log(3, "bx1200_read: %s" % result)
        return result


def run_cmdline():
    global _verbose_level
    global _version 
    ap = argparse.ArgumentParser(description="BX1200 Antennentuner Kontrollprogramm. Version: %s" % _version,
                                 epilog="""Die Frequenzautomatik wird durch dieses Programm
                                           ausgeschaltet muss bei der Verwendung mit dem
                                           Originalprogramm von DL1SNG ggf. wieder aktiviert
                                           werden.
                                        """)
    ap.add_argument("--serial", required=True, help="Serielle Schnittstelle für den BX 1200")
    ap.add_argument("--verbose", type=int, default=0, choices=range(0,4), help="Ausgabenumfang (3=debug)")
    ap.add_argument("--frequency", type=float, help="Setzt die Frequenz in Hz (z.B. 3.5e6) und beendet sich")
    ap.add_argument("--trxclient",
                    help="""rigctld-Adresse <ip/hostname>[<:port>]
                            Frequenzsteuerungsautomatik durch das Funkgeraet.
                            Setzt die Frequenz auf die aktuelle Frequenz des
                            ueber die HAMLIB angeschlossenen Funkgeraets.
                            Die Frequenz wird einmal pro Sekunde abgefragt.
                            Kann zusammen mit --monitor verwenden werden.
                            """)
    ap.add_argument("--monitor",
                    action='store_true',
                    help="""Gibt die Leistung, SWR, Eingangswiderstand und Wirkungsgrad,
                            waehrend des Sendens aus.
                         """)
    ap.add_argument("--beep",
                    action="store_true",
                    help="""Gibt im Monitormodus einen Beep bei einem SWR groesser als 3 aus.
                         """)
    ap.add_argument("--loop",
                    action="store_true",
                    help="""Lauft in einer Schleife im trxclient modus.
                            Versucht eine gestoerte Verbindung zum BX-1200
                            oder zum rigtld wieder aufzunehmen.
                            Sinnvoll wenn das Programm auf einen Raspi dauerhaft
                            laeuft und rigctld bzw. Tuner seperat gestartet bzw.
                            eingeschaltet werden.
                           """)

    args = ap.parse_args()
    _verbose_level = args.verbose
    exit_code = 0
    try:
        if args.frequency is not None:
            exit_code = set_frequency(args)
        elif args.trxclient:
            exit_code = trx_client(args)
        elif args.monitor:
            exit_code = monitor(args)
        else:
            exit_code = -4
    except Exception, e:
        show_exception(e)
        exit_code = -3
    sys.exit(exit_code)


def monitor(args):
    """
    Monitor power and status from BX-1200 untill CTRL c is pressed
    """
    bx_ctrl = BX1200Ctrl(args.serial)
    try:
        bx_ctrl.connect()
    except serial.serialutil.SerialException, e:
        show_exception(e)
        return -1
    sys.stdout.write("Strg C zum Abbrechen\n")
    try:
        while (True):
            bx_ctrl.send_over()
            for result in bx_ctrl.get_tx_data_iterator():
                _monitor_output(args, result)
    except KeyboardInterrupt:
        pass
    return 0


def set_frequency(args):
    """
    sets frequncy and exit
    """
    
    result = None
    exit_code = 0
    try:
        with BX1200Ctrl(args.serial) as bx_ctrl:
            result = bx_ctrl.set_freqency(args.frequency, True)            
    except (serial.serialutil.SerialException, CommunicationError,BX1200Error), e:
        show_exception(e)
        exit_code = -1
    if result:
        log(1, result.to_string())
    return exit_code


def _monitor_output(args, result):
    if result.wrn > 0:
        show_error("LEISTUNG ZU HOCH!")
    swr = calc_swr(result.rin, result.xin)
    if swr >  3.0:
        if args.beep:
            sys.stderr.write("\a\a\a")
        sys.stderr.write("SWR groesser 3\n")
    if args.monitor:
            msg = "Leistung: %.1f W SWR: %.2f (R: %s X: %s) Eff.: %s QRG: %.0f khz" % (result.fwd, swr, result.rin, result.xin, result.eta, result.frq)
            sys.stdout.write("%s\n" % msg)


def trx_client(args):
    """
    Handling running as rigctld client
    Before setting frequency check for transmitting
    Only set new frequency if not in transmit mode
    
    """

    def khz(f):
        return int(f/1000.0)

    first = True
    sys.stdout.write("Strg C zum Abbrechen\n")
    while (args.loop or first):
        last_f = -1
        first = False
        try:
            with BX1200Ctrl(args.serial) as bx_ctrl:
                with HamLibConnection(args.trxclient) as hamlib:
                    log(1, "Verbindung hergestellt")
                    while (1):
                        bx_ctrl.send_over()
                        for result in bx_ctrl.get_tx_data_iterator():
                            _monitor_output(args, result)
                        # we are not in tx_mode
                        # query frequency and change if necessary                         
                        f = hamlib.read_frequency()
                        if f is not None:
                            if khz(f)!= khz(last_f):
                                log(0 if args.monitor else 0, "Neue Frequenz: %.3f kHz" % ( f / 1.e3,))
                                try:
                                    result = bx_ctrl.set_freqency(f)
                                except ValueError:
                                    pass
                                    continue
                                if result:
                                    log(2, result.to_string())
                                    last_f = f
                                else:
                                    show_error("Frequenz %s Hz ist ungueltig" % f)
                                    raise BX1200Error("Error setting frequncy")
                        elif args.loop:                    
                            raise CommunicationError("Keine Frequenz erhalten")                 
        except (serial.serialutil.SerialException, BX1200Error, CommunicationError) as  e:
            show_exception(e)
            if not args.loop:
                return -2
            else:
                conn_err = True
        except KeyboardInterrupt, e:
            return 0
        except Exception, e:
            show_exception(e)
            if not args.loop:
                raise e
        if args.loop:
            sleep_time = 5
            log(1, "Verbindungsfehler. Versuche Reconnect in %ss" % sleep_time)
            time.sleep(sleep_time)
    return 0


if __name__  == "__main__":
    run_cmdline()
